/*
disk2d64 (C)1999 Michael Steil

PC64 transfer routines 17.,18.08.1996
C64 routines           31.12.1995 - 11.08.1996
PC program             11.,12.,13.,14.09.1996
C/Linux port           24.,25.08.1999
*/

#include <stdio.h>
#include <sys/io.h>
#include <time.h>
#include <unistd.h>

#define PROGRAMBUFSIZE 5000 /* max size of disk2d64.prg */
#define TIMEOUT 10 /* seconds */

#define COPYRIGHT "%s v%s (C) 1995-1999 Michael Steil\n"
#define USAGE \
	"Usage: %s [OPTION]\n" \
	"Copy a 1541 disk from a C64 connected to the parallel port through a\n" \
	"PC64 cable to the local computer and write it to standard output.\n" \
	"\n" \
	"  -V       only print version number and exit\n" \
	"  -h       display this help and exit\n" \
	"  -p PORT  use the specified parallel port (default: 0x378)\n" \
	"  -q       quiet operation\n" \
	"  -v       increase verbosity level by one\n" \
	"\n" \
	"Report bugs to mist@c64.org\n"

char sectors[] = { 0,
	21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,21,
	19,19,19,19,19,19,19,
	18,18,18,18,18,18,
	17,17,17,17,17 };
char PRGNAME[] = "disk2d64";
char VERSION[] = "0.0.1";
enum { ERR_OK = 0, ERR_TIMEOUT, ERR_PRGFILE, ERR_IOPERM };
char ERRTEXT[4][40] = { "", "Timeout", "Transfer program file read error", "ioperm() failed" };

int portbase, portin, portout;
int verbosity;


void help()
/* print the usage text and exit successfully */
{
	fprintf(stderr, USAGE, PRGNAME);
	exit(ERR_OK);
}

void error(int errcode)
/* print and error message and exit with error code */
{
	fprintf(stderr, "%s: %s\n", PRGNAME, ERRTEXT[errcode]);
	exit(errcode);
}

/**********************************/
/* General Parallel Port Routines */
/**********************************/

void initport()
/* set the parallel port to a default value */
{
	outb(128, portout);
}

void delay()
/* transition of NMI line from 1 to 0 must not be too fast, so there */
/* has to be a short delay */
{
	int i;
	for (i=0; i<2; ++i) inb(portin);
}

unsigned char getvalue(unsigned char c)
/* wait for clock == 0 (c = 0) or clock == 1 (c = 128) and return the value */
/* from the parallel port */
/* terminate with an error message if C64 doesn't respond within TIMEOUT secs */
{
	unsigned char b;
	clock_t i;

	if (verbosity >= 4) fprintf(stderr, "  waiting for clock == %i...\n", c >> 7);
	i = clock()+TIMEOUT*CLOCKS_PER_SEC;
	while (((b = inb(portin)) & 128) != c) if (clock() > i) error(ERR_TIMEOUT);
	return b;
}

void setvalue(unsigned char b)
/* send a nibble and generate an NMI on the C64 */
/* an NMI is generated by a transition of bit #7 from 1 to 0 */
{
	outb(128, portout); /* prepare NMI */
	delay();
	if (verbosity >= 4) fprintf(stderr, "  send NMI (nibble 0x%x)...\n",b);
	outb(b, portout);	
}

/**********************************/
/* Byte Transfer                  */
/**********************************/

void sendbyte(unsigned char b)
/* transfer a byte to the C64 */
/* terminate with an error message if unsuccessful */
{
	if (verbosity >= 3) fprintf(stderr, " sending byte 0x%x\n", b);
	getvalue(0);
	setvalue(b & 15);
	getvalue(128);
	setvalue(b >> 4);
}

unsigned char getbyte()
/* receive a byte from the C64 */
/* terminate with an error message if unsuccessful */
{
	unsigned char b;

	if (verbosity >= 3) fprintf(stderr, " receiving byte\n");
	setvalue(0); /* signal ready to receive nibble */
	b = (getvalue(0) >> 3) & 15;
	setvalue(0); /* signal ready to receive nibble */
	b = (((getvalue(128) << 1) & 240) | b) ^ 3;
	if (verbosity >= 3) fprintf(stderr, " received byte 0x%x\n", b);
	return b;
}

/**********************************/
/* Block Transfer                 */
/**********************************/

void senddata(unsigned char *buf, int bytes)
/* transfer several bytes to the C64 */
/* terminate with an error message if unsuccessful */
{
	int i;
	if (verbosity >= 2) fprintf(stderr, "sending %i bytes\n", bytes);
	for (i=0; i<bytes; i++) sendbyte(buf[i]);
}

void getdata(unsigned char *buf, int n)
/* receive several bytes from the C64 */
/* terminate with an error message if unsuccessful */
{
	int i;
	if (verbosity >= 2) fprintf(stderr, "receiving %i bytes\n", n);
	for (i=0; i<n; i++)	buf[i] = getbyte();
}

/**********************************/
/* High Level Routines            */
/**********************************/

void transferprg(char *s)
/* transfer the PRG file to the C64 */
{
	char buf[PROGRAMBUFSIZE];
	FILE *in;
	int startaddress, programsize, bytes, i;

	if ((in = fopen(s, "rb")) == NULL) error(ERR_PRGFILE);
	fseek(in, 0L, SEEK_SET); rewind(in);

	startaddress = fgetc(in) | fgetc(in) << 8;
	programsize = fread(&buf, 1, PROGRAMBUFSIZE, in);
	for (i=0; i < programsize; i += 256) {
		bytes = (programsize - i < 256) ? programsize - i : 256;
		sendbyte(0);
		sendbyte(bytes & 255);
		sendbyte((startaddress+i) & 255); sendbyte((startaddress+i)>>8);
		senddata(&buf[i], bytes);
	}
}

void sys(int address)
/* send command to start a program at the given address */
{
	if (verbosity >= 2) fprintf(stderr, "sys %i\n",address);
	sendbyte(1); /* command */
	sendbyte(address & 255);
	sendbyte(address >> 8);
	sendbyte(0); /* X */
	sendbyte(0); /* Y */
	sendbyte(0); /* A */
}

/**********************************/
/* Main                           */
/**********************************/

int main(int argc, char *argv[])
{
	int track;
	char buf[sectors[1]<<8];
	int i,j,sl;

	verbosity = 1;
	portbase = 0x378;

	/* parse command line */
	if (argc>1) for (i=1; i<argc; i++) {
		if (argv[i][0]) if (argv[i][0]=='-') {
			sl = strlen(argv[i]);
			if (sl>1) for (j=1; j<sl; j++) {
				switch (argv[i][j]) {
					case 'V' : fprintf(stdout, "%s\n",VERSION); exit();
					case 'p' : if (i==argc-1) help(); else { portbase=strtol(argv[i+1],NULL,0); argv[i+1][0]=0; } break;
					case 'q' : verbosity = 0; break;
					case 'v' : verbosity++; break;
					default : help();
				}
			}
		} else help();
	}

	if (verbosity) fprintf(stderr, COPYRIGHT, PRGNAME, VERSION);
	if (verbosity) fprintf(stderr, "Initializing...\n");

	portout = portbase; portin = portbase + 1;
	/* permit direct I/O access */
	if (ioperm(portbase, 3, 1)) error(ERR_IOPERM);
	initport();

	transferprg("disk2d64.prg");
	sys(0x1000);

	if (verbosity) fprintf(stderr, "Transfer...\n");

	for (track=1;track<=35;track++) {
		if (verbosity) fprintf(stderr, "Track %i\r", track);
		getdata(buf, sectors[track]<<8);
		fwrite(&buf, 256, sectors[track], stdout);
	}

	if (verbosity) fprintf(stderr, "\nDone.\n");

	return ERR_OK;
}

